#perl
#==============================================================================
#
#  Faktories und Mocks
#
#  Demo-Programm
#
#  Die Aufgabe besteht darin, die Anwendung des Readers nicht zu veraendern.
#  Der folgende Code soll unveraendert funktionieren:
#
#  my $example = new WordReader();
#  $example->CountWordsOfWebFile($remote_file, $minimalCount);
#
#==============================================================================

use warnings;
use strict;

$|=1;

use Benchmark;

use WordCounter1;
use WordCounter2;
use WordCounter3;
use WordCounter4;
use WordCounter5;

use Factory6;
use WordCounter6;

use Factory7;

# Example 8 und 9
use WordCounter;
use DiContainer;

#==============================================================================
#
#  package main
#
#==============================================================================
package main;

# our $remote_file = "http://www.bla.de/blablabla.html";
# our $remote_file = "http://www.jupiter-programs.de/fun_or_not_fun/die_Bruecke.html";
# our $remote_file = "http://en.wikipedia.org/wiki/Main_Page";
# our $remote_file = "http://de.wikipedia.org/wiki/Perl_(Programmiersprache)";
our $remote_file = "http://search.cpan.org/~jv/Getopt-Long-2.38/lib/Getopt/Long.pm";

our $minimalCount = 30;
our @zeitMessung;

# fuer Example 6 und 7
our $Factory = undef;

# fuer Example 8 und 9
our $DiContainer = new DiContainer();    

#------------------------------------------------------------------------------
#  Den DiContainer zurueckliefern
#------------------------------------------------------------------------------
sub GetDiContainer
{
    my $caller = shift;
    my $callerClass = ref ($caller)  ||  $caller;

    return $DiContainer;
}

#------------------------------------------------------------------------------
#  Beispiel 1,
#
#  der Reader liest selbst aus dem Web und analysiert in einer Methode
#------------------------------------------------------------------------------
sub Example1
{
    &PrintExample("Beispiel 1, der Reader liest selbst aus dem Web \n"
		  ."und analysiert in einer Methode.");
    
    # Normaler Ablauf, kein Test
    my $example1 = new WordCounter1();
    $example1->CountWordsOfWebFile($remote_file, $minimalCount);

    print "\n\n";
}

#------------------------------------------------------------------------------
#  Beispiel 2,
#
#  der Reader liest selbst aus dem Web und analysiert in zweiter Methode
#------------------------------------------------------------------------------
sub Example2
{
    &PrintExample("Beispiel 2, der Reader liest selbst aus dem Web\n"
		  ."und analysiert in zweiter Methode.");

    # Normaler Ablauf, kein Test
    my $example2 = new WordCounter2();
    $example2->CountWordsOfWebFile($remote_file, $minimalCount);

    print "\n\n";
}

#------------------------------------------------------------------------------
#  Beispiel 3,
#
#  der Reader liest selbst aus dem Web mit einer dritten Methode 
#  und analysiert in zweiter Methode
#------------------------------------------------------------------------------
sub Example3
{
    &PrintExample("Beispiel 3, der Reader liest selbst aus dem Web mit einer dritten Methode\n"
                 ."und analysiert in zweiter Methode.");

    # Normaler Ablauf, kein Test
    my $example3 = new WordCounter3();
    $example3->CountWordsOfWebFile($remote_file, $minimalCount);

    print "\n\n";
}

#------------------------------------------------------------------------------
#  Beispiel 4,
#
#  der Reader laesst einen Web-Reader aus dem Web lesen,
#  den er aber selbst mit new erzeugt. 
#  Die Analyse fuehrt er dann selbst durch, in einer zweiten Methode
#------------------------------------------------------------------------------
sub Example4
{
    &PrintExample("Beispiel 4, der Reader laesst einen Web-Reader aus dem Web lesen,\n"
                 ."den er aber selbst mit new erzeugt.\n"
                 ."Die Analyse fuehrt er dann selbst durch, in einer zweiten Methode.");

    # Normaler Ablauf, kein Test
    my $example4 = new WordCounter4();
    $example4->CountWordsOfWebFile($remote_file, $minimalCount);

    print "\n\n";
}

#------------------------------------------------------------------------------
#  Beispiel 5,
#
#  der Reader laesst einen Web-Reader aus dem Web lesen,
#  den er von einer Factory erzeugen laesst.
#  Die Factory erzeugt er noch selbst.
#  Die Analyse fuehrt er dann selbst durch, in einer zweiten Methode
#------------------------------------------------------------------------------
sub Example5
{
    &PrintExample("Beispiel 5, der Reader laesst einen Web-Reader aus dem Web lesen,\n"
                 ."den er von einer Factory erzeugen laesst.\n"
                 ."Die Factory erzeugt er noch selbst.\n"
                 ."Die Analyse fuehrt er dann selbst durch, in einer zweiten Methode.");

    # Normaler Ablauf, kein Test
    my $example5 = new WordCounter5();
    $example5->CountWordsOfWebFile($remote_file, $minimalCount);

    print "\n\n";
}

#------------------------------------------------------------------------------
#  Beispiel 6,
#
#  der Reader laesst einen Web-Reader aus dem Web lesen,
#  den er von einer Factory erzeugen laesst.
#  Die Factory erzeugt er nicht mehr selbst, er verwendet die globale Factory.
#  Die Analyse fuehrt er dann selbst durch, in einer zweiten Methode
#------------------------------------------------------------------------------
sub Example6
{
    &PrintExample("Beispiel 6, der Reader laesst einen Web-Reader aus dem Web lesen,\n"
                 ."den er von einer Factory erzeugen laesst.\n"
                 ."Die Factory erzeugt er nicht mehr selbst, er verwendet die globale Factory.\n"
                 ."Die Analyse fuehrt er dann selbst durch, in einer zweiten Methode.");

    # Factory belegen
    $WordCounter6::WebReaderFactory = new Factory6();
    
    # Normaler Ablauf, kein Test
    my $example6 = new WordCounter6();
    $example6->CountWordsOfWebFile($remote_file, $minimalCount);

    print "\n\n";
}

#------------------------------------------------------------------------------
#  Beispiel 7,
#
#  Es muss keine neue Reader-Klasse erstellt werden,
#  WordCounter ist ausreichend.
#  Nur die Factory muss angepasst werden.
#  Der Reader laesst einen Web-Reader aus dem Web lesen (denkt er !)
#  den er von einer Factory erzeugen laesst.
#  Die Factory erzeugt er nicht mehr selbst, er verwendet die globale Factory.
#  Er erhaelt einen Stub statt der Original-Klasse.
#  Die Analyse fuehrt er dann selbst durch, in einer zweiten Methode.
#------------------------------------------------------------------------------
sub Example7
{
    &PrintExample("Beispiel 7,\n"
                 ."Es muss keine neue Reader-Klasse erstellt werden,\n"
                 ."WordCounter6 ist ausreichend.\n"
                 ."Nur die Factory muss angepasst werden.\n"
                 ."Der Reader laesst einen Web-Reader aus dem Web lesen (denkt er !)\n"
                 ."den er von einer Factory erzeugen laesst.\n"
                 ."Die Factory erzeugt er nicht mehr selbst, er verwendet die globale Factory.\n"
                 ."Er erhaelt einen Stub statt der Original-Klasse.\n"
                 ."Die Analyse fuehrt er dann selbst durch, in einer zweiten Methode.\n"
        );

    # nur noch die Factory anpassen!
    $WordCounter6::WebReaderFactory = new Factory7(); # Stub statt WebReader liefern
    
    # Test mit Stub
    my $example7 = new WordCounter6();
    $example7->CountWordsOfWebFile($remote_file, $minimalCount);

    print "\n\n";
}

#------------------------------------------------------------------------------
#  Beispiel 8,
#
#  Jedesmal eine Factory zu erstellen und zu verteilen, ist muehselig.
#  Einfacher geht es mit DI-Containern statt Factories.
#  Dazu muss der WordCounter ein letztes Mal angepasst werden.
#------------------------------------------------------------------------------
sub Example8
{
    &PrintExample("Beispiel 8,\n"
                 ."Jedesmal eine Factory zu erstellen und zu verteilen, ist muehselig.\n"
                 ."Einfacher geht es mit DI-Containern statt Factories.\n"
                 ."Dazu muss der WordCounter ein letztes Mal angepasst werden.\n"
        );

    # Den DiContainer NICHT fuellen!
    
    # Normaler Ablauf, kein Test
    my $example8 = new WordCounter();
    $example8->CountWordsOfWebFile($remote_file, $minimalCount);

    print "\n\n";
}

#------------------------------------------------------------------------------
#  Beispiel 9,
#
#  Und jetzt einfach ueber ein Closure etwas anderes als WebReader hinterlegen
#  durch $DiContainer->DepositForCreate(...)
#------------------------------------------------------------------------------
sub Example9
{
    &PrintExample("Beispiel 9,\n"
                  ."Und jetzt einfach ueber ein Closure etwas anderes als WebReader hinterlegen\n"
                  ."durch \$DiContainer->DepositForCreate(...)"
        );

    # nur noch den DiContainer fuellen!
    
    $DiContainer->DepositForCreate
        (new WordCounter(),                  # fuer welche Klasse hinterlegt
         "IWebReader",                      # Schnittstelle des zurueckgelieferten Objekts
         sub { return new WebReaderStub() } # Stub erzeugen ueber Closure
        );

    # Test mit Stub
    my $example9 = new WordCounter();
    $example9->CountWordsOfWebFile($remote_file, $minimalCount);

    print "\n\n";
}

#------------------------------------------------------------------------------
#
#   Beispiel ausfhren und Zeit stoppen
#
#------------------------------------------------------------------------------
sub RunExample
{
    my $exampleCode = shift;

    my $timeLauf1 = new Benchmark;

    eval $exampleCode;

    $timeLauf1 = timediff (new Benchmark, $timeLauf1);

    my $zeit = "$exampleCode - " . timestr($timeLauf1) . "\n";
    push (@zeitMessung, $zeit);

    # print "$zeit\nPause, press <return>\n";
    # <STDIN>;
}

#------------------------------------------------------------------------------
#
#   Beispielueberschrift ausgeben
#
#------------------------------------------------------------------------------
sub PrintExample
{
    my $text = shift;

    $text =~ s/\n/\n#  /og;

    print "#------------------------------------------------------------------------------\n";
    print "#  $text\n";
    print "#------------------------------------------------------------------------------\n";
}

#==============================================================================
#
#  Main
#
#==============================================================================

print "\n\n";

&RunExample('&Example1(); # WebReader');
&RunExample('&Example2(); # WebReader');
&RunExample('&Example3(); # WebReader');
&RunExample('&Example4(); # WebReader');
&RunExample('&Example5(); # WebReader');

&RunExample('&Example6(); # WebReader');
&RunExample('&Example7(); # Stub     ');
&RunExample('&Example8(); # WebReader');
&RunExample('&Example9(); # Stub     ');

print "\n--- Zeitvergleich --------------------------------------------------\n\n";
print join ('', @zeitMessung);
